/*  -*- pse-c -*-
 *----------------------------------------------------------------------------
 * Filename: iegd_interface.c
 * $Revision: 1.1.2.5 $
 *----------------------------------------------------------------------------
 * Gart and DRM driver for Intel Embedded Graphics Driver
 * Copyright © 2007, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 
 * 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include <linux/pagemap.h>
#include "global.h"
#include "intelpci.h"
#include "interface_abs.h"
#include "igd_abs.h"

static struct agp_memory *alloc_agpphysmem_i8xx(
	size_t pg_count, int type);
static void *i8xx_alloc_pages(size_t pg_count,
	unsigned int order);
static void i8xx_destroy_pages(void *addr,
	size_t pg_count, unsigned int order);

dispatch_table_t driver_dispatch_list[] = {
	{ PCI_DEVICE_ID_810,   &drv_alm },
	{ PCI_DEVICE_ID_810DC, &drv_alm },
	{ PCI_DEVICE_ID_810E,  &drv_alm },
	{ PCI_DEVICE_ID_815,   &drv_alm },
	{ PCI_DEVICE_ID_830M,  &drv_alm },
	{ PCI_DEVICE_ID_845G,  &drv_alm },
	{ PCI_DEVICE_ID_855,   &drv_alm },
	{ PCI_DEVICE_ID_865G,  &drv_alm },
	{ PCI_DEVICE_ID_915GD, &drv_nap },
	{ PCI_DEVICE_ID_915AL, &drv_nap },
	{ PCI_DEVICE_ID_945G,  &drv_nap },
	{ PCI_DEVICE_ID_945GM, &drv_nap },
	{ PCI_DEVICE_ID_945GME,&drv_nap },
	{ PCI_DEVICE_ID_Q35,   &drv_nap },
	{ PCI_DEVICE_ID_965G,  &drv_gn4 },
	{ PCI_DEVICE_ID_946GZ, &drv_gn4 },
	{ PCI_DEVICE_ID_G965,  &drv_gn4 },
	{ PCI_DEVICE_ID_Q965,  &drv_gn4 },
	{ PCI_DEVICE_ID_GM965, &drv_gn4 },
	{ PCI_DEVICE_ID_GME965,&drv_gn4 },
	{ 0, NULL },
};

/* Structure contained bit mask for the page table entries */
struct gatt_mask iegd_cmn_masks[] =
{
	{.mask = I810_PTE_VALID, .type = 0},
	{.mask = (I810_PTE_VALID | I810_PTE_LOCAL),
		.type = AGP_DCACHE_MEMORY},
	{.mask = I810_PTE_VALID, .type = 0}
};


int iegd_cmn_configure(void)
{
	struct aper_size_info_fixed *current_size;
	u32 temp;
	u16 gmch_ctrl;
	int i;

	AGN_DEBUG("Enter");

	current_size = A_SIZE_FIX(agp_bridge->current_size);

	pci_read_config_dword(private_data.pdev,
		I915_GMADDR, &temp);

	agp_bridge->gart_bus_addr = (temp & PCI_BASE_ADDRESS_MEM_MASK);

	if(private_data.pdev->device != PCI_DEVICE_ID_Q35) {
		pci_read_config_word(agp_bridge->dev,I830_GMCH_CTRL,&gmch_ctrl);
		gmch_ctrl |= I830_GMCH_ENABLED;
		pci_write_config_word(agp_bridge->dev,I830_GMCH_CTRL,gmch_ctrl);

		global_cache_flush();
		agp_bridge->driver->tlb_flush(0);

		writel(agp_bridge->gatt_bus_addr|I810_PGETBL_ENABLED,
				private_data.registers+I810_PGETBL_CTL);
		/* PCI Posting. */
		readl(private_data.registers+I810_PGETBL_CTL);
	}

	if (agp_bridge->driver->needs_scratch_page) {
		for (i = private_data.gtt_entries;
			i < current_size->num_entries; i++) {
			writel(agp_bridge->scratch_page, private_data.gtt+i);
			readl(private_data.gtt+i);	/* PCI Posting. */
		}
	}
	global_cache_flush();

	AGN_DEBUG("Exit");

	return 0;
}

void iegd_cmn_init_gtt_entries(void)
{
	u16 gmch_ctrl;
	u32 iegd_scratch, iegd_scratch2;
	int gtt_entries;
	u8 rdct;
	int local = 0;
	static const int ddt[4] = { 0, 16, 32, 64 };
	int size;
	int gtt_enabled = FALSE;

	AGN_DEBUG("Enter");

	pci_read_config_word(agp_bridge->dev,I830_GMCH_CTRL,&gmch_ctrl);

	gtt_enabled = readl(private_data.registers + I810_PGETBL_CTL) &
		I810_PGETBL_ENABLED;

	/* A note on stolen memory:
	 *  Intel chipsets set aside a small area at the top of system memory
	 *  for VGA framebuffers etc. When the Intel device is the VGA
	 *  device, this memory is used to contain the GTT itself, and a scratch
	 *  memory page. Therefore the actual available memory already populated
	 *  in the GTT is the stolen memory minus the 4k scratch page minus the
	 *  128 page table.
	 *
	 *  Additionally, the embedded firmware may further alter this amount.
	 *  It can either allocate additional memory to be placed in the GTT
	 *  or use some stolen memory for data. If the IEGD vBIOS has altered
	 *  the amount we can detect it by reading a well-defined scratch
	 *  register.
	 *
	 *  When the Intel Graphics Device is not the VGA device, i.e.
	 *  the system boots with a PCI card, then this driver discards
	 *  the stolen memory.
	 *
	 * We obtain the size of the GTT, which is also stored (for some
	 * reason) at the top of stolen memory. Then we add 4KB to that
	 * for the video BIOS popup, which is also stored in there. */

	size = agp_bridge->driver->fetch_size() + 4;
	AGN_DEBUG("Size from fetch size + 4 = %x", size);

	if (agp_bridge->dev->device == PCI_DEVICE_ID_INTEL_82830_HB ||
	    agp_bridge->dev->device == PCI_DEVICE_ID_INTEL_82845G_HB) {
		switch (gmch_ctrl & I830_GMCH_GMS_MASK) {
		case I830_GMCH_GMS_STOLEN_512:
			gtt_entries = KB(512) - KB(size);
			break;
		case I830_GMCH_GMS_STOLEN_1024:
			gtt_entries = MB(1) - KB(size);
			break;
		case I830_GMCH_GMS_STOLEN_8192:
			gtt_entries = MB(8) - KB(size);
			break;
		case I830_GMCH_GMS_LOCAL:
			rdct = readb(private_data.registers+I830_RDRAM_CHANNEL_TYPE);
			gtt_entries = (I830_RDRAM_ND(rdct) + 1) *
					MB(ddt[I830_RDRAM_DDT(rdct)]);
			local = 1;
			break;
		default:
			gtt_entries = 0;
			break;
		}
	} else {
		switch (gmch_ctrl & I830_GMCH_GMS_MASK) {
		case I855_GMCH_GMS_STOLEN_1M:
			gtt_entries = MB(1) - KB(size);
			break;
		case I855_GMCH_GMS_STOLEN_4M:
			gtt_entries = MB(4) - KB(size);
			break;
		case I855_GMCH_GMS_STOLEN_8M:
			gtt_entries = MB(8) - KB(size);
			break;
		case I855_GMCH_GMS_STOLEN_16M:
			gtt_entries = MB(16) - KB(size);
			break;
		case I855_GMCH_GMS_STOLEN_32M:
			gtt_entries = MB(32) - KB(size);
			break;
		case I915_GMCH_GMS_STOLEN_48M:
			/* Check it's really I915G */
			if (agp_bridge->dev->device == PCI_DEVICE_ID_BRIDGE_915GD      ||
			    agp_bridge->dev->device == PCI_DEVICE_ID_BRIDGE_915AL      ||
			    agp_bridge->dev->device == PCI_DEVICE_ID_BRIDGE_945G       ||
			    agp_bridge->dev->device == PCI_DEVICE_ID_BRIDGE_945GM      ||
			    agp_bridge->dev->device == PCI_DEVICE_ID_BRIDGE_945GME     ||
			    agp_bridge->dev->device == PCI_DEVICE_ID_BRIDGE_965G       ||
			    agp_bridge->dev->device == PCI_DEVICE_ID_BRIDGE_G965       ||
			    agp_bridge->dev->device == PCI_DEVICE_ID_BRIDGE_Q965       ||
			    agp_bridge->dev->device == PCI_DEVICE_ID_BRIDGE_GM965	   ||
			    agp_bridge->dev->device == PCI_DEVICE_ID_BRIDGE_GME965	   ||
			    agp_bridge->dev->device == PCI_DEVICE_ID_BRIDGE_946GZ      )
				gtt_entries = MB(48) - KB(size);
			else
				gtt_entries = 0;
			break;
		case I915_GMCH_GMS_STOLEN_64M:
			/* Check it's really I915G */
			if (agp_bridge->dev->device == PCI_DEVICE_ID_BRIDGE_915GD      ||
			    agp_bridge->dev->device == PCI_DEVICE_ID_BRIDGE_915AL      ||
			    agp_bridge->dev->device == PCI_DEVICE_ID_BRIDGE_945G       ||
			    agp_bridge->dev->device == PCI_DEVICE_ID_BRIDGE_945GM      ||
			    agp_bridge->dev->device == PCI_DEVICE_ID_BRIDGE_945GME     ||
			    agp_bridge->dev->device == PCI_DEVICE_ID_BRIDGE_965G       ||
			    agp_bridge->dev->device == PCI_DEVICE_ID_BRIDGE_G965       ||
			    agp_bridge->dev->device == PCI_DEVICE_ID_BRIDGE_Q965       ||
			    agp_bridge->dev->device == PCI_DEVICE_ID_BRIDGE_GM965	   ||
			    agp_bridge->dev->device == PCI_DEVICE_ID_BRIDGE_GME965	   ||
			    agp_bridge->dev->device == PCI_DEVICE_ID_BRIDGE_946GZ      )
				gtt_entries = MB(64) - KB(size);
			else
				gtt_entries = 0;
		default:
			gtt_entries = 0;
			break;
		}
	}

	/* if GTT is not enabled, then initialize gtt entries to 0 */
	if (!gtt_enabled) {
		AGN_DEBUG("GTT is disabled");
		AGN_LOG("IGD not primary, throwing away stolen memory.");

		/* Update the scratch registers to say that we have no stolen memory */
		writel((0xE1DF << 16), private_data.registers + 0x71410);

		iegd_scratch = readl(private_data.registers + 0x71410);
		iegd_scratch |= 0x4;

		writel(iegd_scratch, private_data.registers + 0x71410);

		/* say that we have 0 stolen memory regardless of what was
		 * really in there */
		writel(0, private_data.registers + 0x71418);

		gtt_entries = 0;
	}

	iegd_scratch = readl(private_data.registers + 0x71410);

	if(((iegd_scratch>>16) == 0xE1DF) && (iegd_scratch & 0x4)) {
		AGN_LOG("IEGD Firmware Detected");
		/* IEGD firmware found, and Mem Reservation Flag present */
		iegd_scratch2 = readl(private_data.registers + 0x71418);
		gtt_entries = (iegd_scratch2 & 0xFFFF) * 4096;
	}

	if (gtt_entries > 0)
		AGN_LOG("Detected %dK %s memory.",
		       gtt_entries / KB(1), local ? "local" : "stolen");
	else
		AGN_LOG("No pre-allocated video memory detected.\n");

	gtt_entries /= KB(4);
	private_data.gtt_entries = gtt_entries;

	AGN_DEBUG("Exit");
}

int AGP_FREE_GATT(iegd_cmn_free_gatt_table)
{
	AGN_DEBUG("Enter");
	return 0;
	AGN_DEBUG("Exit");
}

void AGP_ENABLE(iegd_cmn_agp_enable)
{
	AGN_DEBUG("Enter");
	return;
	AGN_DEBUG("Exit");
}

struct agp_memory *iegd_cmn_alloc_by_type(
	size_t pg_count, int type)
{
	struct agp_memory *new;

	AGN_DEBUG("Enter");

	/* AGP_DCACHE_MEMORY use by certain chipset only, especially
	 * chipset from almador family. */
	if(private_data.pdev->device == PCI_DEVICE_ID_810   ||
	   private_data.pdev->device == PCI_DEVICE_ID_810DC ||
	   private_data.pdev->device == PCI_DEVICE_ID_810E  ||
	   private_data.pdev->device == PCI_DEVICE_ID_815) {
		if (type == AGP_DCACHE_MEMORY) {
			if (pg_count != private_data.num_dcache_entries) {
				AGN_ERROR("Page count error");
				AGN_DEBUG("pg_count=%d, num_dcache_entries=%d",
					pg_count, private_data.num_dcache_entries);
				return NULL;
			}

			new = agp_create_memory(1);
			if (new == NULL) {
				AGN_ERROR("Allocating memory failed");
				return NULL;
			}

			new->type = AGP_DCACHE_MEMORY;
			new->page_count = pg_count;
			new->num_scratch_pages = 0;
			vfree(new->memory);
			AGN_DEBUG("AGP_DCACHE_MEMORY.. Exit");
			return new;
		}
	}

	if (type == AGP_PHYS_MEMORY) {
		AGN_DEBUG("AGP_PHYS_MEMORY.. Exit");
		return alloc_agpphysmem_i8xx(pg_count, type);
	}

	AGN_DEBUG("NULL.. Exit");
	return NULL;
}

void iegd_cmn_free_by_type(struct agp_memory *curr)
{
	unsigned int order;

	AGN_DEBUG("Enter");

	switch (curr->page_count) {
	case 1:
		order = 0;   /* pg_count = 1 => 2 ^ 0 */
		break;
	case 4:
		order = 2;   /* pg_count = 4 => 2 ^ 2 */
		break;
	case 8:
		order = 3;   /* pg_count = 8 => 2 ^ 3 */
		break;
	default:
		/* This case should never happen */
		return;
	}

	agp_free_key(curr->key);
	if(curr->type == AGP_PHYS_MEMORY) {
		i8xx_destroy_pages(gart_to_virt(curr->memory[0]), curr->page_count,
			order);
		IGD_FREE_MEM(curr);
	}
	kfree(curr);

	AGN_DEBUG("Exit");
}

static struct agp_memory *alloc_agpphysmem_i8xx(size_t pg_count, int type)
{
	struct agp_memory *new;
	void *addr;
	unsigned int order, i;

	AGN_DEBUG("Enter");

	/* To support RGBA hardware cursor which may require contiguous physical
	 * memory to be allocated with either 1, 4 or 8 pages. 8 pages is
	 * the worst case for 830 which requires 4 pages and 4 page alignment.
	 */
	switch (pg_count) {
	case 1:
		order = 0;   /* pg_count = 1 => 2 ^ 0 */
		break;
	case 4:
		order = 2;   /* pg_count = 4 => 2 ^ 2 */
		break;
	case 8:
		order = 3;   /* pg_count = 8 => 2 ^ 3 */
		break;
	default:
		return NULL;
	}

	addr = i8xx_alloc_pages(pg_count, order);
	if (addr == NULL) {
		AGN_ERROR("Allocating pages failed");
		return NULL;
	}

	new = agp_create_memory(pg_count);
	if (new == NULL) {
		AGN_ERROR("Allocating memory failed");
		return NULL;
	}

	new->memory[0] = virt_to_gart(addr);
	for (i = 1; i < pg_count; i++) {
		new->memory[i] = new->memory[i-1] + PAGE_SIZE;
	}
	new->page_count = pg_count;
	new->num_scratch_pages = pg_count;
	new->type = AGP_PHYS_MEMORY;
	new->physical = new->memory[0];

	AGN_DEBUG("Exit");
	return new;
}

static void *i8xx_alloc_pages(size_t pg_count, unsigned int order)
{
	struct page * page;

	AGN_DEBUG("Enter");

	page = alloc_pages(GFP_KERNEL, order);
	if (page == NULL) {
		AGN_ERROR("Allocating kernel page failed");
		return NULL;
	}

	if (change_page_attr(page, pg_count, PAGE_KERNEL_NOCACHE) < 0) {
		global_flush_tlb();
		__free_page(page);
		AGN_ERROR("Change page attribute failed");
		return NULL;
	}
	global_flush_tlb();
	get_page(page);
	SetPageLocked(page);
	atomic_inc(&agp_bridge->current_memory_agp);
	return page_address(page);

	AGN_DEBUG("Exit");
}

static void i8xx_destroy_pages(void *addr,
	size_t pg_count, unsigned int order)
{
	struct page *page;

	AGN_DEBUG("Enter");

	if (addr == NULL)
		return;

	page = virt_to_page(addr);
	change_page_attr(page, pg_count, PAGE_KERNEL);
	global_flush_tlb();
	put_page(page);
	unlock_page(page);
	free_pages((unsigned long)addr, order);
	atomic_dec(&agp_bridge->current_memory_agp);

	AGN_DEBUG("Exit");
}

unsigned long AGP_MASK_MEMORY(iegd_cmn_mask_memory)
{
	struct agp_bridge_data *brdg = AGP_BRIDGE_VAR;

	AGN_DEBUG("Enter");
	/* Type checking must be done elsewhere */
	return addr | AGP_MASK_ADDR(brdg);
	AGN_DEBUG("Exit");
}

int iegd_cmn_insert_entries(struct agp_memory *mem,
	off_t pg_start, int type)
{
	int i,j,num_entries;
	void *temp;

	AGN_DEBUG("Enter");

	temp = agp_bridge->current_size;
	num_entries = A_SIZE_FIX(temp)->num_entries;

	if (pg_start < private_data.gtt_entries) {
		AGN_ERROR("Trying to insert into local/stolen memory");
		AGN_DEBUG("pg_start == 0x%.8lx,private_data.gtt_entries =="
			"%d", pg_start,private_data.gtt_entries);
		return -EINVAL;
	}

	/* If we try to write beyond gtt table, return error */
	if ((pg_start + mem->page_count) > num_entries) {
		AGN_ERROR("Trying to write beyond aperture limit");
		AGN_DEBUG("pg_start=0x%.8lx, mem->page_count=%d,"
				"num_entries=%d", pg_start, mem->page_count,
				num_entries);
		return -EINVAL;
	}

	/* The i830 can't check the GTT for entries since its read only,
	 * depend on the caller to make the correct offset decisions.
	 */

	if ((type != 0 && type != AGP_PHYS_MEMORY) ||
		(mem->type != 0 && mem->type != AGP_PHYS_MEMORY)) {
		AGN_ERROR("Unsupported memory type");
		AGN_DEBUG("mem->type=%x, type=%x", mem->type, type);
		return -EINVAL;
	}

	global_cache_flush();
	agp_bridge->driver->tlb_flush(mem);

	for (i = 0, j = pg_start; i < mem->page_count; i++, j++) {
		writel(AGP_MASK_GTT(), private_data.gtt+j);
		readl(private_data.gtt+j);	/* PCI Posting. */
	}

	global_cache_flush();
	agp_bridge->driver->tlb_flush(mem);

	AGN_DEBUG("Exit");

	return 0;
}

int iegd_cmn_remove_entries(struct agp_memory *mem,
	off_t pg_start, int type)
{
	int i;

	AGN_DEBUG("Enter");

	global_cache_flush();
	agp_bridge->driver->tlb_flush(mem);

	if (pg_start < private_data.gtt_entries) {
		AGN_ERROR("Trying to disable local/stolen memory");
		AGN_DEBUG("pg_start=0x%.8lx, private_data.gtt_entries=%d",
				pg_start, private_data.gtt_entries);
		return -EINVAL;
	}

	for (i = pg_start; i < (mem->page_count + pg_start); i++) {
		writel(agp_bridge->scratch_page, private_data.gtt+i);
		readl(private_data.gtt+i);
	}

	global_cache_flush();
	agp_bridge->driver->tlb_flush(mem);

	AGN_DEBUG("Exit");

	return 0;
}
